/*
 * Copyright (c) 2024 Sascha Wildner <swildner@gmail.com>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <6502.h>
#include <c64.h>
#include <stdio.h>

#define	__unused		__attribute__((__unused__))

#define	_CB2CRT_STATUS		(*(volatile char *)0x90)
#define	_CB2CRT_TIME0		(*(volatile char *)0xa0)
#define	_CB2CRT_TIME1		(*(volatile char *)0xa1)
#define	_CB2CRT_TIME2		(*(volatile char *)0xa2)
#define	_CB2CRT_PNTR		(*(volatile char *)0xd3)
#define	_CB2CRT_USRADD_LO	(*(volatile char *)0x311)
#define	_CB2CRT_USRADD_HI	(*(volatile char *)0x312)

#define	_CB2CRT_LEFT_D(src, tmp, n)	\
			_cb2crt_lmr_d((src), (tmp), 1, (n), false)
#define	_CB2CRT_MID_D(src, tmp, s, n)	\
			_cb2crt_lmr_d((src), (tmp), (s), (n), false)
#define	_CB2CRT_RIGHT_D(src, tmp, n)	\
			_cb2crt_lmr_d((src), (tmp), 1, (n), true)

static __unused void _cb2crt_delay(int32_t);
static __unused char _cb2crt_get_st(void);
static __unused int32_t _cb2crt_get_ti(void);
static __unused char *_cb2crt_get_ti_d(char *);
static __unused void _cb2crt_set_clock_from_ti_d(char *);
static __unused char *_cb2crt_cwp(char, char, char *, char *, bool, bool);
static __unused char *_cb2crt_lmr_d(const char *, char *, char, char, bool);
static __unused int32_t _cb2crt_power(int32_t, int32_t);
static __unused void _cb2crt_print_comma(void);
static __unused void _cb2crt_print_signed_int(int32_t);
static __unused void _cb2crt_spc(char);
static __unused char *_cb2crt_str_d(int32_t);
static __unused void _cb2crt_sys(unsigned);
static __unused void _cb2crt_tab(char);
static __unused int32_t _cb2crt_usr(int32_t);

static __unused void _set_ti_d(char *, uint32_t);

typedef int32_t (*usr_func_t)(int32_t);
typedef void (*sys_func_t)(void);

static void
_cb2crt_delay(int32_t jiffies)
{
	int32_t i;
	char r = VIC.rasterline;

	for (i = 0; i < jiffies; i++) {
		while (VIC.rasterline == r) { };
		while (VIC.rasterline != r) { };
	}
}

static char
_cb2crt_get_st(void)
{
	return _CB2CRT_STATUS;
}

static int32_t
_cb2crt_get_ti(void)
{
	return _CB2CRT_TIME0 * 65536UL + _CB2CRT_TIME1 * 256UL + _CB2CRT_TIME2;
}

static char *
_cb2crt_get_ti_d(char *ti_d)
{
	int32_t secs = _cb2crt_get_ti() / 60;
	uint32_t clk;

	clk = secs / 3600 * 10000;
	secs %= 3600;
	clk += secs / 60 * 100;
	secs %= 60;
	clk += secs;
	_set_ti_d(ti_d, clk);
	return ti_d;
}

static void
_cb2crt_set_clock_from_ti_d(char *ti_d)
{
	uint32_t clk, clk2, jiffies;

	clk = clk2 = atol(ti_d);
	if (clk >= 774021)
		clk -= 774021;
	else if (clk > 235959)
		clk = 0;
	if (clk % 100 > 59)
		clk += 40;
	if (clk / 100 % 100 > 59)
		clk += 4000;
	if (clk != clk2)
		_set_ti_d(ti_d, clk);
	jiffies = clk / 10000 * 3600;
	clk %= 10000;
	jiffies += clk / 100 * 60;
	clk %= 100;
	jiffies += clk;
	jiffies *= 60;
	SEI();
	_CB2CRT_TIME0 = jiffies / 65536;
	jiffies %= 65536;
	_CB2CRT_TIME1 = jiffies / 256;
	jiffies %= 256;
	_CB2CRT_TIME2 = jiffies;
	CLI();
}

static void
_set_ti_d(char *ti_d, uint32_t clk)
{
	char i = 5;

	memset(ti_d, '0', 6);
	while (clk > 0) {
		ti_d[i--] = (char)(clk % 10) + '0';
		clk /= 10;
	}
}

/* concat with plus */
static char *
_cb2crt_cwp(char c1, char c2, char *s1, char *s2, bool l_ischar, bool m_ischar)
{
	static char str[256];
	char len;

	if (l_ischar && m_ischar) {
		str[0] = c1;
		str[1] = c2;
		str[2] = '\0';
	} else if (!l_ischar && m_ischar) {
		len = (char)strlen(s1);
		strcpy(str, s1);
		str[len] = c1;
		str[len + 1] = '\0';
	} else if (l_ischar && !m_ischar) {
		str[0] = c1;
		strcpy(&str[1], s1);
	} else if (!l_ischar && !m_ischar) {
		strcpy(str, s1);
		strcat(str, s2);
	}
	return str;
}

/* {LEFT,MID,RIGHT}$ */
static char *
_cb2crt_lmr_d(const char *src, char *tmp, char start, char len, bool right)
{
	char loclen;

	start--;
	loclen = (char)strlen(src);
	if (len > 0 && loclen > 0) {
		if (right)
			start = (loclen - len);
		if ((start + len) <= loclen) {
			memcpy(tmp, src + start, len);
			tmp[len] = '\0';
		} else {
			return NULL;
		}
	} else {
		tmp[0] = '\0';
	}
	return tmp;
}

/* ^ */
static int32_t
_cb2crt_power(int32_t base, int32_t power)
{
	int32_t res = 1;

	while (power > 0) {
		if (power % 2)
			res *= base;
		base *= base;
		power /= 2;
	}
	return res;
}

static void
_cb2crt_print_comma(void)
{
	char d = 10 - (_CB2CRT_PNTR % 10);

	while (d-- != 0)
		putchar(CH_CURS_RIGHT);
}

static void
_cb2crt_print_signed_int(int32_t num)
{
	char digits[11], idx = 0;
	signed char i;
	uint32_t unum;

	putchar(' ');
	if (num == 0) {
		putchar('0');
	} else if (num == INT32_MIN) {
		putchar('-');
		putchar('2');
		putchar('1');
		putchar('4');
		putchar('7');
		putchar('4');
		putchar('8');
		putchar('3');
		putchar('6');
		putchar('4');
		putchar('8');
	} else {
		if (num < 0) {
			putchar('-');
			unum = -num;
		} else {
			unum = num;
		}
		while (unum > 0) {
			digits[idx++] = (char)(unum % 10) + '0';
			unum /= 10;
		}
		for (i = idx - 1; i >= 0; i--)
			putchar(digits[i]);
	}
	putchar(' ');
}

static void
_cb2crt_spc(char len)
{
	while (len-- != 0)
		putchar(CH_CURS_RIGHT);
}

static char *
_cb2crt_str_d(int32_t num)
{
	static char str[12];
	char digits[12], idx = 0, si = 0;
	signed char i;
	uint32_t unum;

	if (num == 0) {
		str[si++] = '0';
	} else if (num == INT32_MIN) {
		str[si++] = '-';
		str[si++] = '2';
		str[si++] = '1';
		str[si++] = '4';
		str[si++] = '7';
		str[si++] = '4';
		str[si++] = '8';
		str[si++] = '3';
		str[si++] = '6';
		str[si++] = '4';
		str[si++] = '8';
	} else {
		if (num < 0) {
			str[si++] = '-';
			unum = -num;
		} else {
			unum = num;
		}
		while (unum > 0) {
			digits[idx++] = (char)(unum % 10) + '0';
			unum /= 10;
		}
		for (i = idx - 1; i >= 0; i--)
			str[si++] = digits[i];
	}
	str[si] = '\0';
	return str;
}

static void
_cb2crt_sys(unsigned arg)
{
	sys_func_t _cb2crt_sys_func = (sys_func_t)arg;

	_cb2crt_sys_func();
}

static void
_cb2crt_tab(char npos)
{
	npos -= _CB2CRT_PNTR;

	while (npos-- != 0)
		putchar(CH_CURS_RIGHT);
}

static int32_t
_cb2crt_usr(int32_t arg)
{
	usr_func_t _cb2crt_usr_func =
	    (usr_func_t)(_CB2CRT_USRADD_HI * 256 + _CB2CRT_USRADD_LO);

	return _cb2crt_usr_func(arg);
}
