/***************************************************************/
/*                                                             */
/*  if_sc92031.c                                               */
/*                                                             */
/*  Silan Ethernet Netcard Divice Driver                       */
/*                                                             */
/***************************************************************/

#include <sys/param.h>          /* defines used in kernel.h */
#include <sys/systm.h>          /* printf */
#include <sys/sockio.h>
#include <sys/mbuf.h>
#include <sys/malloc.h>
#include <sys/kernel.h>         /* types used in module initial intialization */
#include <sys/socket.h>
#include <sys/endian.h>

#include <net/if.h>
#include <net/if_var.h>
#include <net/if_arp.h>
#include <net/ethernet.h>
#include <net/if_dl.h>
#include <net/if_media.h>

#include <net/bpf.h>

#include <vm/vm.h>              /* for vtophys */
#include <vm/pmap.h>            /* for vtophys */
#include <machine/clock.h>      /* for DELAY */
#include <machine/bus_pio.h>
#include <machine/bus_memio.h>
#include <machine/bus.h>
#include <machine/resource.h>
#include <sys/bus.h>
#include <sys/rman.h>

#include <pci/pcireg.h>
#include <pci/pcivar.h>

#include "pci/if_sc92031reg.h"

/* Default to using PIO access for netcard driver*/
#define SL_USEIOSPACE  
#undef  SILAN_DEBUG

#ifndef lint
static const char slcardid[] =
"$FreeBSD-4.7: /usr/src/sys/pci/silan.c,v 1.0 2003/01/10  gaoyonghong$";
#endif


#ifdef SILAN_DEBUG
#define PDEBUG(args...)   printf(const char* ...)
#else
#define PDEBUG(args...)
#endif

static struct silan_pcitype silan_dev[] = {
	{ SL_VENDORID, SL_DEVICEID, "silan 10/100M ethernet netcard" },
	{ 0, 0, NULL }
};


static int silan_probe		__P((device_t));
static int silan_attach		__P((device_t));
static int silan_detach		__P((device_t));
static void silan_shutdown	__P((device_t));
static int silan_suspend        __P((device_t));
static int silan_resume         __P((device_t));

static void silan_reset         __P((struct silan_softc *));
static void silan_init		__P((void *));

static void silan_tx            __P((struct ifnet *));
static void silan_rx		__P((struct silan_softc *));
static void silan_tx_intr	__P((struct silan_softc *));
static void silan_media_intr    __P((struct silan_softc *));
static void silan_interrupt	__P((void *));
static int  silan_ioctl		__P((struct ifnet *, unsigned long, caddr_t));
static void silan_stop		__P((struct silan_softc *));
static void silan_watchdog	__P((struct ifnet *));

static int silan_media_upd      __P((struct ifnet *));

static void silan_media_stat    __P((struct ifnet *, struct ifmediareq *));
static void silan_mii_cmd       __P((struct silan_softc *, u_int32_t, unsigned long *));
static void silan_media_cfg     __P((struct silan_softc *));
static void silan_mac_cfg       __P((struct silan_softc *));
static u_int32_t silan_ether_crc32  __P((caddr_t));
static void silan_set_multi	__P((struct silan_softc *));
static void silan_init_tx	__P((struct silan_softc *));
static void silan_tick          __P((void *)); 



#ifdef SL_USEIOSPACE
#define  SL_RID    SL_PCI_IOAD
#define  SL_RES    SYS_RES_IOPORT
#else
#define  SL_RID     SL_PCI_MEMAD
#define  SL_RES     SYS_RES_MEMORY
#endif


static device_method_t  silan_methods[]={
	/* Device interface */
	DEVMETHOD(device_probe,		silan_probe),
	DEVMETHOD(device_attach,	silan_attach),
	DEVMETHOD(device_detach,	silan_detach),
	DEVMETHOD(device_shutdown,	silan_shutdown),
	DEVMETHOD(device_suspend,       silan_suspend),
        DEVMETHOD(device_resume,        silan_resume),

	/* bus interface */
	/*DEVMETHOD(bus_print_child,	bus_generic_print_child),*/
	DEVMETHOD(bus_driver_added,	bus_generic_driver_added),
	
	{ 0, 0 }
};

static driver_t  silan_driver = {
	"sln",
	silan_methods,
	sizeof(struct silan_softc)
};

static devclass_t  silan_devclass;

DRIVER_MODULE(silan, pci, silan_driver, silan_devclass, 0, 0);


/* Probe for silan netcard chip. Check the PCI vendor and device
 * IDs  and return a device name if we find a match.
 */
static int silan_probe(dev)	
  struct device    *dev;
{
	struct silan_pcitype	*adapter;
        
        adapter = silan_dev;
                        	
	while (adapter->silan_name != NULL) {
	    if ((pci_get_vendor(dev) == adapter->silan_vendor) 
		     && (pci_get_device(dev) == adapter->silan_device)){
		       	device_set_desc(dev, adapter->silan_name);
			printf("pci_vendor = %x, pci_device = %x \n", adapter->silan_vendor,adapter->silan_device);
			return(0);
            }
	       
            adapter++;	
	}
        
	return(ENXIO);
}


/* the chip reset */
static void silan_reset(adapter)
     struct silan_softc  *adapter;
{
        SILAN_WRITE_4(adapter, SL_CFG0,SL_SOFT_RESET);
        DELAY(200000);
        SILAN_WRITE_4(adapter, SL_CFG0, 0x0);
        DELAY(10000); 
        
        return;
} 


/* Attach the interface. Allocate softc structures */
static int silan_attach(dev)
   device_t    dev;
{
        int			s;
	unsigned char		eaddr[ETHER_ADDR_LEN];
	u_int32_t		command;

	struct silan_softc	*adapter;
	struct ifnet		*ifp;
	int			unit;
	int                     rid;
	int                     error = 0;      
        
        s = splimp();

	adapter = device_get_softc(dev);
	unit = device_get_unit(dev);
	bzero(adapter, sizeof(struct silan_softc));

        /* configure pci command */
	command = pci_read_config(dev, SL_PCI_COMMAND, 4);
	
	if ((command & (SL_CMD_BUSMASTER | SL_CMD_MEMORY |SL_CMD_IO)) != 0x7) {
	     command |= (SL_CMD_BUSMASTER | SL_CMD_MEMORY |SL_CMD_IO);
	     pci_write_config(dev, SL_PCI_COMMAND, command, 4);
	}
	
	command = pci_read_config(dev, SL_PCI_COMMAND, 4);
	
#ifdef SL_USEIOSPACE
       if (!(command & SL_CMD_IO)) {
	   printf("sln%d: failed to enable I/O ports!\n", unit);
	   error = ENXIO;
	   goto fail;
       }
#else
       if (!(command & SL_CMD_MEMORY)) {
           printf("sln%d: failed to enable memory mapping!\n", unit);
	   error = ENXIO;
	   goto fail;
       }
#endif

       rid = SL_RID;
       adapter->silan_res = bus_alloc_resource(dev, SL_RES, &rid, 0, ~0, 1, RF_ACTIVE);
                  
       if (adapter->silan_res == NULL) {
	   printf ("sln%d: couldn't map ports/memory\n", unit);
	   error = ENXIO;
	   goto fail;
      }

      adapter->silan_bustag = rman_get_bustag(adapter->silan_res);
      adapter->silan_bushandle = rman_get_bushandle(adapter->silan_res);
      
      /* alloc pci irq */
      rid = 0;
      adapter->silan_irq = bus_alloc_resource(dev, SYS_RES_IRQ, &rid, 0, ~0, 
                                              1,RF_SHAREABLE | RF_ACTIVE);

      if (adapter->silan_irq == NULL) {
	  printf("sln%d: couldn't map interrupt\n", unit);
	  bus_release_resource(dev, SL_RES, SL_RID, adapter->silan_res);
	  error = ENXIO;
	  goto fail;
      }

       error = bus_setup_intr(dev, adapter->silan_irq, INTR_TYPE_NET,
                    silan_interrupt, adapter, &adapter->silan_intrhand);

      if (error) {
	  bus_release_resource(dev, SYS_RES_IRQ, 0, adapter->silan_irq);
	  bus_release_resource(dev, SL_RES, SL_RID, adapter->silan_res);
	  printf("sln%d: couldn't set up irq\n", unit);
	  goto fail;
      }
     
      /* Get MAC address */
     ((u_int32_t *)(&eaddr))[0] = be32toh(SILAN_READ_4(adapter,SL_MAC_ADDR0));
     ((u_int16_t *)(&eaddr))[2] = be16toh(SILAN_READ_4(adapter,SL_MAC_ADDR1)); 
  	
      printf("sln%d: Ethernet address:%x:%x:%x:%x:%x:%x\n",
	         unit,eaddr[0],eaddr[1],eaddr[2],eaddr[3],eaddr[4],eaddr[5]);
	
      adapter->silan_unit = unit;
      bcopy(eaddr, (caddr_t)&adapter->arpcom.ac_enaddr, ETHER_ADDR_LEN);
     
      /* alloc rx buffer space */
      adapter->silan_bufdata.silan_rx_buf = (char *)contigmalloc(SL_RX_BUFLEN, 
                                             M_DEVBUF, M_NOWAIT, 0, 0xffffffff, PAGE_SIZE, 0);

      if (adapter->silan_bufdata.silan_rx_buf == NULL) {
	     printf("sln%d: no memory for rx buffers!\n", adapter->silan_unit);
	     bus_teardown_intr(dev, adapter->silan_irq, adapter->silan_intrhand);
	     bus_release_resource(dev, SYS_RES_IRQ, 0, adapter->silan_irq);
	     bus_release_resource(dev, SL_RES, SL_RID, adapter->silan_res);		
	     error = ENXIO;
	     goto fail;
       }
	
      callout_handle_init(&adapter->silan_state);
		
       ifp = &adapter->arpcom.ac_if;          /* network-visible interface */
       ifp->if_softc = adapter;               /* pointer to driver state */
       ifp->if_unit = unit;
       ifp->if_name = "sln";
       ifp->if_mtu = ETHERMTU;                 /* maximum transmission unit */
       ifp->if_flags = IFF_BROADCAST | IFF_SIMPLEX | IFF_MULTICAST;
       ifp->if_timer = 0;                      /* time 'till if_watchdog called */
       ifp->if_init = silan_init;              /* init routine */
       ifp->if_start = silan_tx;               /* initiate output routine(tx routine) */
       ifp->if_ioctl = silan_ioctl;            /* ioctl routine */
       ifp->if_output = ether_output;          /* output routine (enqueue) */
       ifp->if_watchdog = silan_watchdog;      /* timer routine */
       ifp->if_baudrate = 100000000;           /* linespeed */
       ifp->if_snd.ifq_maxlen = IFQ_MAXLEN;    /* output queue */
       
       /* initial media */
       ifmedia_init(&adapter->ifmedia, 0, silan_media_upd, silan_media_stat);

       /*supported media types */
       ifmedia_add(&adapter->ifmedia,IFM_ETHER|IFM_AUTO, 0, NULL);
       ifmedia_add(&adapter->ifmedia, IFM_ETHER|IFM_10_T, 0, NULL); 
       ifmedia_add(&adapter->ifmedia, IFM_ETHER|IFM_10_T|IFM_HDX, 0, NULL);
       ifmedia_add(&adapter->ifmedia, IFM_ETHER|IFM_10_T|IFM_FDX, 0, NULL);
       ifmedia_add(&adapter->ifmedia, IFM_ETHER|IFM_100_TX, 0, NULL);
       ifmedia_add(&adapter->ifmedia, IFM_ETHER|IFM_100_TX|IFM_HDX, 0, NULL);
       ifmedia_add(&adapter->ifmedia, IFM_ETHER|IFM_100_TX|IFM_FDX, 0, NULL);
      
       /* Choose a default media. */
       ifmedia_set(&adapter->ifmedia, IFM_ETHER | IFM_AUTO);
                   
       ether_ifattach(ifp, ETHER_BPF_SUPPORTED);

       splx(s);

       return(0);
fail:
       splx(s);
       return(error);
}



/* Stop the adapter and free any mbufs allocated to the RX and TX buffers */
static void silan_stop(adapter)		
   struct silan_softc	  *adapter;
{
	int		i;
	struct ifnet	*ifp;
	u_int32_t       intr_status;
        
        ifp = &adapter->arpcom.ac_if;
	ifp->if_timer = 0;

        untimeout(silan_tick, adapter, adapter->silan_state);

        /* disable Tx/Rx */
        adapter->txcfg &= ~SL_TXCFG_EN;
        adapter->rxcfg &= ~SL_RXCFG_EN;
	SILAN_WRITE_4(adapter, SL_TX_CONFIG, adapter->txcfg);
	SILAN_WRITE_4(adapter, SL_RX_CONFIG, adapter->rxcfg);

        /* Clear interrupt */
	SILAN_WRITE_4(adapter, SL_INT_MASK, 0);
	intr_status = SILAN_READ_4(adapter, SL_INT_STATUS);

	/* Free the TX list buffers */
	for (i = 0; i < SL_TXD_CNT; i++){
	       if (adapter->silan_bufdata.silan_tx_buf[i] != NULL) {
		    m_freem(adapter->silan_bufdata.silan_tx_buf[i]);
		    adapter->silan_bufdata.silan_tx_buf[i] = NULL;
	            SILAN_WRITE_4(adapter, SL_TSAD0 + i*4, 0);
	       }
	}

	ifp->if_flags &= ~(IFF_RUNNING | IFF_OACTIVE);  /* resources allocated/transmission in progress */

	return;
}



static int silan_detach(dev)
     device_t   dev;
{
	struct silan_softc	*adapter;
	struct ifnet		*ifp;
	int			s;

	s = splimp();

        adapter = (struct silan_softc *)device_get_softc(dev);
	ifp = &adapter->arpcom.ac_if;

	ether_ifdetach(ifp, ETHER_BPF_SUPPORTED);
	silan_stop(adapter);

	bus_generic_detach(dev);
	
	bus_teardown_intr(dev, adapter->silan_irq, adapter->silan_intrhand);
	bus_release_resource(dev, SYS_RES_IRQ, 0, adapter->silan_irq);
	bus_release_resource(dev, SL_RES, SL_RID, adapter->silan_res);
	
        contigfree(adapter->silan_bufdata.silan_rx_buf,SL_RX_BUFLEN, M_DEVBUF);
	
	splx(s);
        
	return(0);
}


static int silan_media_upd(ifp)
     struct ifnet     *ifp;
{
        struct silan_softc   *adapter;
        struct ifmedia       *ifm;

        adapter = ifp->if_softc;
        ifm = &adapter->ifmedia;

        if (IFM_TYPE(ifm->ifm_media) != IFM_ETHER)
            return(EINVAL);
        
        if (ifp->if_flags & IFF_UP){
	    silan_init(adapter);
        }
	
        return 0;
}


static void silan_media_stat(ifp,ifmr)
     struct ifnet        *ifp;
     struct ifmediareq   *ifmr;
{
        struct silan_softc  *adapter;
        unsigned long       phys[2];   
        u_int32_t           temp;
     
        adapter = ifp->if_softc;
        ifmr->ifm_status = IFM_AVALID;
        ifmr->ifm_active = IFM_ETHER;
         
        phys[0] = SL_MII_STAT;
        silan_mii_cmd(adapter, SL_MII0_READ, phys);
         
        if (phys[1] & SL_MIISTAT_LINK)
	    ifmr->ifm_status |= IFM_ACTIVE;
                 
        temp = SILAN_READ_4(adapter,SL_PHY_CTRL);

        if ((temp & (SL_PHYCTL_DUX|SL_PHYCTL_SPD100 |SL_PHYCTL_SPD10))== 0x60800000)  
           ifmr->ifm_active |= IFM_AUTO;
           else if ((temp & (SL_PHYCTL_DUX|SL_PHYCTL_SPD100))== 0x40800000)
                  ifmr->ifm_active |= IFM_100_TX |IFM_FDX;
                 else if ((temp & SL_PHYCTL_SPD100) == 0x40000000)
                         ifmr->ifm_active |= IFM_100_TX |IFM_HDX;   
                      else if ((temp & (SL_PHYCTL_DUX|SL_PHYCTL_SPD10))== 0x20800000)
                              ifmr->ifm_active |= IFM_10_T |IFM_FDX;
                           else if ((temp & SL_PHYCTL_SPD10)== 0x20000000)
		                   ifmr->ifm_active|= IFM_10_T |IFM_HDX;       

	 silan_mii_cmd(adapter,SL_MII0_SCAN, phys);
         return;
}


            
/* command selected in MII command register  */
static void silan_mii_cmd(adapter, cmd, phys)
     struct silan_softc   *adapter;
     u_int32_t            cmd;
     unsigned long        *phys;
{   	
        u_int32_t            mii_status;
         
        SILAN_WRITE_4(adapter, SL_MII_CMD0, SL_MII0_DIVEDER);

       do {
          mii_status = 0;
          DELAY(10);
          mii_status = SILAN_READ_4(adapter,SL_MII_STATUS);
       } while (mii_status & SL_MIISTAT_BUSY);

       switch (cmd) {
       case SL_MII0_SCAN:
         SILAN_WRITE_4(adapter, SL_MII_CMD1, 0x1 << 6);
         SILAN_WRITE_4(adapter, SL_MII_CMD0, SL_MII0_DIVEDER | SL_MII0_SCAN);
         break;
	     
       case SL_MII0_READ:
         SILAN_WRITE_4(adapter, SL_MII_CMD1, phys[0] << 6);   
         SILAN_WRITE_4(adapter, SL_MII_CMD0, SL_MII0_DIVEDER | SL_MII0_READ);
         break;

       default: /* WRITE */  
         SILAN_WRITE_4(adapter, SL_MII_CMD1, phys[0] << 6| phys[1] << 11); 
         SILAN_WRITE_4(adapter, SL_MII_CMD0, SL_MII0_DIVEDER | SL_MII0_WRITE );
         break;
       }

       do {
         DELAY(10); 
         mii_status = SILAN_READ_4(adapter,SL_MII_STATUS);
       } while (mii_status & SL_MIISTAT_BUSY);
       
       if (SL_MII0_READ == cmd) {
          phys[1] = (mii_status >> 13) & 0xffff;
       }
             
}


/* Set media speed and duplex mode */
static void  silan_media_cfg(adapter)
  struct silan_softc	*adapter;
{   
       unsigned long           phys[2];
       u_int32_t               mediatype;
       u_int32_t               temp;

       mediatype = (&adapter->ifmedia)->ifm_cur->ifm_media;
       
       temp = SILAN_READ_4(adapter,SL_PHY_CTRL);
       temp &= ~(SL_PHYCTL_DUX|SL_PHYCTL_SPD100 |SL_PHYCTL_SPD10);
       temp |= (SL_PHYCTL_ANE |SL_PHYCTL_RESET);
      
       /************************************************/
       /*  currently set media word by selected media  */
       /*                                              */   
       /*  IFM_ETHER = 0x00000020                      */
       /*  IFM_AUTO=0, IFM_10_T=3,  IFM_100_TX=6       */
       /*  IFM_FDX=0x00100000    IFM_HDX=0x00200000    */
       /************************************************/ 
       switch (mediatype) {
          case 0x00000020:  
	    PDEBUG(" autoselet supported\n");
	    temp |= (SL_PHYCTL_DUX|SL_PHYCTL_SPD100 |SL_PHYCTL_SPD10);
	    adapter->ifmedia.ifm_media = IFM_ETHER|IFM_AUTO;
            ifmedia_set(&adapter->ifmedia, IFM_ETHER | IFM_AUTO);
            break;
          case 0x23:         
          case 0x00200023: 
	    PDEBUG(" 10Mbps half_duplex supported\n");
    	    temp |= SL_PHYCTL_SPD10;
            adapter->ifmedia.ifm_media = IFM_ETHER|IFM_10_T|IFM_HDX;
            ifmedia_set(&adapter->ifmedia, IFM_ETHER | IFM_10_T |IFM_HDX);
            break;
    
          case 0x00100023:
	    PDEBUG("10Mbps full_duplex supported\n");
            temp |= (SL_PHYCTL_SPD10 |SL_PHYCTL_DUX);
	    adapter->ifmedia.ifm_media = IFM_ETHER |IFM_10_T|IFM_FDX;
            ifmedia_set(&adapter->ifmedia, IFM_ETHER | IFM_10_T | IFM_FDX);
            break;
  
	  case 0x26:
          case 0x00200026:
	    PDEBUG("100Mbps half_duplex supported\n");
  	    temp |= SL_PHYCTL_SPD100;
            adapter->ifmedia.ifm_media = IFM_ETHER|IFM_100_TX|IFM_HDX;
            ifmedia_set(&adapter->ifmedia, IFM_ETHER | IFM_100_TX |IFM_HDX);
            break;

          case 0x00100026:
	    PDEBUG("100Mbps full_duplex supported\n");
            temp |= (SL_PHYCTL_SPD100 |SL_PHYCTL_DUX);
	    adapter->ifmedia.ifm_media = IFM_ETHER|IFM_100_TX|IFM_FDX;
            ifmedia_set(&adapter->ifmedia, IFM_ETHER | IFM_100_TX |IFM_FDX);
            break;
            
          default:
            break;
       }
       
       SILAN_WRITE_4(adapter,SL_PHY_CTRL, temp);
       
       DELAY(10000);
       temp &= ~SL_PHYCTL_RESET;
       SILAN_WRITE_4(adapter,SL_PHY_CTRL, temp);
       
       DELAY(1000);
       phys[0] = SL_MII_JAB;
       phys[1] = SL_PHY_16_JAB_ENB |SL_PHY_16_PORT_ENB;
       silan_mii_cmd(adapter, SL_MII0_WRITE, phys);
       
       adapter->connect = false;
       silan_mii_cmd(adapter, SL_MII0_SCAN, phys);
}

     
static void silan_mac_cfg(adapter)
     struct silan_softc	*adapter;
{
       struct ifnet            *ifp;
       unsigned long           flowcfg = 0;

       ifp = &adapter->arpcom.ac_if;

       /* Set the initial TX/RX/Flow Control configuration */
       adapter->rxcfg = SL_RXCFG_LOW_THRESHOLD | SL_RXCFG_HIGH_THRESHOLD;
       adapter->txcfg = TX_CFG_DEFAULT;

       if (adapter->txenablepad)
           adapter->txcfg |= 0x20000000;
         
       if (adapter->media_speed == IFM_10_T)
           adapter->txcfg |= SL_TXCFG_DATARATE;

       if (adapter->media_duplex == IFM_FDX) {
         adapter->rxcfg |= SL_RXCFG_FULLDX;
         adapter->txcfg |= SL_TXCFG_FULLDX;
         flowcfg = SL_FLOWCTL_FULLDX | SL_FLOWCTL_EN;
       } else {
         adapter->rxcfg &= ~SL_RXCFG_FULLDX;
         adapter->txcfg &= ~SL_TXCFG_FULLDX;
       }

       /* if promiscuous mode, set the allframes bit. */
       if (ifp->if_flags & IFF_PROMISC)
          adapter->rxcfg |= (SL_RXCFG_EN|SL_RXCFG_RCV_SMALL|SL_RXCFG_RCV_HUGE|SL_RXCFG_RCV_ERR|SL_RXCFG_RCV_BROAD|SL_RXCFG_RCV_MULTI|SL_RXCFG_RCV_ALL);
       else
          adapter->rxcfg &= ~(SL_RXCFG_EN|SL_RXCFG_RCV_SMALL|SL_RXCFG_RCV_HUGE|SL_RXCFG_RCV_ERR||SL_RXCFG_RCV_BROAD|SL_RXCFG_RCV_MULTI|SL_RXCFG_RCV_ALL);
	
       /* Set capture broadcast bit to capture broadcast frames */
       if (ifp->if_flags & IFF_BROADCAST)
          adapter->rxcfg |= SL_RXCFG_EN | SL_RXCFG_RCV_BROAD;
       else
          adapter->rxcfg &= ~(SL_RXCFG_EN | SL_RXCFG_RCV_BROAD);

       /* Program the multicast filter, if necessary */
       silan_set_multi(adapter);
       
       SILAN_WRITE_4(adapter, SL_RX_CONFIG, adapter->rxcfg);
       SILAN_WRITE_4(adapter, SL_TX_CONFIG, adapter->txcfg);
       SILAN_WRITE_4(adapter, SL_FLOW_CTRL, flowcfg);
}

   

static u_char shade_map[ ] = {0x0, 0x8, 0x4, 0xc, 0x2, 0xa, 0x6, 0xe,
                              0x1, 0x9, 0x5, 0xd, 0x3, 0xb, 0x7, 0xf};
                                   
/* Calculate CRC32 of a multicast group address */
static u_int32_t silan_ether_crc32(addr)
     caddr_t      addr;
{
       u_int32_t   crc, crcr;
       int        i, j;
       unsigned char   data = 0;
       /* Compute CRC for the address value. */

       crc = 0xFFFFFFFF; /* initial value */

       for (i = ETHER_ADDR_LEN; i >0; i--) {
          data = *addr++;
                   
          for (j = 0; j < 8; j++) {
    	      if (((data & 0x1)^(crc & 0x1))!= 0){
    	          crc >>= 1;
    	          crc ^= 0xEDB88320;
    	      } else { 
    	          crc >>= 1;
    	      }
    	      data >>= 1;
         }
       }
       
       crcr = shade_map[crc >> 28];
       crcr |= (shade_map[(crc >> 24) & 0xf] << 4);
       crcr |= (shade_map[(crc >> 20) & 0xf] << 8);
       crcr |= (shade_map[(crc >> 16) & 0xf] << 12);
       crcr |= (shade_map[(crc >> 12) & 0xf] << 16);
       crcr |= (shade_map[(crc >> 8) & 0xf] << 20);
       crcr |= (shade_map[(crc >> 4) & 0xf] << 24);
       crcr |= (shade_map[crc & 0xf] << 28);
       
       return crcr;
}



/* Program the 64-bit multicast hash filter */
static void silan_set_multi(adapter)
   struct silan_softc   *adapter;
{
       struct ifnet		*ifp;
       u_int32_t		crc = 0;
       u_int32_t		mc_g[2] = { 0, 0 };
       struct ifmultiaddr	*ifma;
       int			i, j;
               
       ifp = &adapter->arpcom.ac_if;
       if (ifp->if_flags & IFF_PROMISC){
             printf(" Promisc mode is enalbe\n");
             adapter->rxcfg |=  SL_RXCFG_EN | SL_RXCFG_RCV_MULTI ;
	     mc_g[0] = mc_g[1] = 0xFFFFFFFF;
       } else if (ifp->if_flags & IFF_ALLMULTI) {
	         printf(" Allmulti mode is enalbe\n");
                 adapter->rxcfg |=  SL_RXCFG_EN | SL_RXCFG_RCV_MULTI ;
		 mc_g[0] = mc_g[1] = 0xFFFFFFFF;
              } else if (ifp->if_flags & IFF_MULTICAST ){
	                    PDEBUG("Multicast mode is enable\n");   
                            adapter->rxcfg |= SL_RXCFG_EN | SL_RXCFG_RCV_MULTI;

                            /* first, zero all the existing hash bits */
	                   mc_g[0] = mc_g[1] = 0;

                           for (i=0, ifma =(ifp->if_multiaddrs.lh_first); ifma != NULL && (i < ifma->ifma_refcount); i++,ifma =( ifma->ifma_link.le_next)) {
		                 j = 0;
		  
		                 if ((ifma->ifma_addr->sa_family) != AF_LINK)
			                 continue;
			
		                 crc =~ silan_ether_crc32(LLADDR((struct sockaddr_dl *)ifma->ifma_addr));
                                 crc >>= 24 ;
                                 
                                 if (crc & 0x1)   j |= 0x2;
                                 if (crc & 0x2)   j |= 0x1;
                                 if (crc & 0x10)  j |= 0x20;
                                 if (crc & 0x20)  j |= 0x10;
                                 if (crc & 0x40)  j |= 0x8;
                                 if (crc & 0x80)  j |= 0x4;
                                 
                                 if ( j > 31)
                                      mc_g[0] |= (0x1 << (j -32 ));
                                 else
                                      mc_g[1] |= (0x1 << j);
			   }
                     } else {
	                    adapter->rxcfg &= ~(SL_RXCFG_EN | SL_RXCFG_RCV_MULTI);
                     }
               
           SILAN_WRITE_4(adapter, SL_RX_CONFIG, adapter->rxcfg);
	   SILAN_WRITE_4(adapter, SL_MULTI_GROUP0, mc_g[0]);
	   SILAN_WRITE_4(adapter, SL_MULTI_GROUP1, mc_g[1]);
       
       	return;
}


/* Initialize the TX/Rx descriptors */
static void silan_init_tx(adapter)
   struct silan_softc	*adapter;
{
       int   i;
    
       adapter->silan_bufdata.cur_tx = 0;
       adapter->silan_bufdata.dirty_tx = 0;
    	    
       for (i = 0; i < SL_TXD_CNT; i++) {
	  adapter->silan_bufdata.silan_tx_buf[i] = NULL;
	  SILAN_WRITE_4(adapter, SL_TSAD0 + (i * 4), 0);
       }
     
       return;
}
   
   
/* Software & Hardware Initialize */
static void silan_init(x)		
    void	*x;
{
      struct silan_softc      *adapter = x;
      struct ifnet	      *ifp;
      int		      s;
      
      PDEBUG("silan_init\n");
      s = splimp();
      
      ifp = &adapter->arpcom.ac_if;
                
      /*  soft reset the chip */
      silan_reset(adapter);
      
      /* disable interrupt */
      SILAN_WRITE_4(adapter, SL_INT_MASK, 0);
  	  
      /*SILAN_WRITE_4(adapter, SL_MII_CMD0, SL_MII0_DIVEDER);*/

      /* clear multicast address */
      SILAN_WRITE_4(adapter, SL_MULTI_GROUP0, 0);
      SILAN_WRITE_4(adapter, SL_MULTI_GROUP1, 0);

      /* Init the RX buffer start address register. */
      SILAN_WRITE_4(adapter, SL_RBSA, vtophys(adapter->silan_bufdata.silan_rx_buf));
      adapter->silan_bufdata.dirty_rx = vtophys(adapter->silan_bufdata.silan_rx_buf);
      
      /* Init TX descriptors. */
      silan_init_tx(adapter);

      /* configure RX buffer size */
      if (adapter->tx_early_ctrl && adapter->rx_early_ctrl)
               SILAN_WRITE_4(adapter, SL_CFG1, SL_EARLY_RX |SL_EARLY_TX | SL_RXBUF_64 | SL_RXFIFO_1024BYTES);
      else if (adapter->tx_early_ctrl)
                SILAN_WRITE_4(adapter, SL_CFG1, SL_EARLY_TX | SL_RXBUF_64);
           else if (adapter->rx_early_ctrl)
                        SILAN_WRITE_4(adapter, SL_CFG1, SL_EARLY_RX | SL_RXBUF_64 | SL_RXFIFO_1024BYTES);
                else
                        SILAN_WRITE_4(adapter, SL_CFG1, SL_RXBUF_64);

      /*MII media configuration */              
      silan_media_cfg(adapter);
      
      if (adapter->connect) {
         /*Enable transmit and receive */
         adapter->rxcfg |= SL_RXCFG_EN;
         adapter->txcfg |= SL_TXCFG_EN;
      } else {
         adapter->rxcfg &= ~SL_RXCFG_EN;
         adapter->txcfg &= ~SL_TXCFG_EN;
      }

      SILAN_WRITE_4(adapter, SL_TX_CONFIG, adapter->txcfg);
      SILAN_WRITE_4(adapter, SL_RX_CONFIG, adapter->rxcfg);

      /* Enable interrupts */
      SILAN_WRITE_4(adapter, SL_INT_MASK, SL_INRTS);
      
      adapter->suspended = false;
     
     ifp->if_flags |= IFF_RUNNING;
     ifp->if_flags &= ~IFF_OACTIVE;    

     (void)splx(s);
     adapter->silan_state = timeout(silan_tick,adapter,hz);
           
     return;
}



/* Transmit Packet */
static void silan_tx(ifp)	
   struct ifnet		*ifp;
{
	struct silan_softc	*adapter;
	struct mbuf	        *m_head = NULL;
	struct mbuf             *m_new = NULL;
	int                     entry;
      
        adapter = ifp->if_softc;
                      
        if((!adapter->connect) || (ifp->if_flags & IFF_OACTIVE)) {
            return;				
	}	
               
	while (SL_CUR_TXBUF(adapter) == NULL) {/*SL_CUR_TXBUF(x) = x->silan_bufdata.silan_tx_buf[x->silan_bufdata.cur_tx]*/
	      entry = adapter->silan_bufdata.cur_tx;	

              IF_DEQUEUE(&ifp->if_snd, m_head);	
	      if (m_head == NULL)
		   break;
            
	      MGETHDR(m_new, M_DONTWAIT, MT_DATA);
	      if (m_new == NULL) {
		   printf("sln%d: no memory for tx descriptor", adapter->silan_unit);
		   IF_PREPEND(&ifp->if_snd, m_head);	
	           ifp->if_flags |= IFF_OACTIVE;
		   break;
	      }
	
	     if (( m_head->m_pkthdr.len > MHLEN) || ( 60 > MHLEN)) {
               	   MCLGET(m_new, M_DONTWAIT);
		   if (!(m_new->m_flags & M_EXT)) {
			m_freem(m_new);
			printf("sln%d: no memory for tx descriptor",adapter->silan_unit);
			IF_PREPEND(&ifp->if_snd, m_head);	
	                ifp->if_flags |= IFF_OACTIVE;
			break;
		   }
	       }

	      m_copydata(m_head, 0,  m_head->m_pkthdr.len, mtod(m_new, caddr_t));              
	      m_new->m_pkthdr.len = m_new->m_len =  m_head->m_pkthdr.len;
	      m_freem(m_head);
	      m_head = m_new;
              SL_CUR_TXBUF(adapter) = m_head;

             /* if there's a BPF listerner, bounce a copy of this frame to him */
             if (ifp->if_bpf)
		bpf_mtap(ifp,SL_CUR_TXBUF(adapter));
    
             /* Transmit the frame */
	     SILAN_WRITE_4(adapter, ((entry * 4) + SL_TSAD0), 
	                       vtophys(mtod(SL_CUR_TXBUF(adapter), caddr_t)));
	      
             /* calculate length of Transmit the frame */
             if (( SL_CUR_TXBUF(adapter)->m_pkthdr.len < 60) && (!adapter->txenablepad)) {
                     memset(mtod(m_head, char*)+ m_head->m_pkthdr.len, 0x20, 60- m_head->m_pkthdr.len);
		     SILAN_WRITE_4(adapter,(entry * 4) + SL_TSD0 , 60);
             } else if (SL_CUR_TXBUF(adapter)->m_pkthdr.len < 100)
                       SILAN_WRITE_4(adapter, (entry * 4) + SL_TSD0, SL_CUR_TXBUF(adapter)->m_pkthdr.len);
                    else if (SL_CUR_TXBUF(adapter)->m_pkthdr.len<300)
                            SILAN_WRITE_4(adapter, (entry * 4) + SL_TSD0, 0x30000 | SL_CUR_TXBUF(adapter)->m_pkthdr.len );
                          else
                            SILAN_WRITE_4(adapter, (entry * 4) + SL_TSD0, 0x50000 | SL_CUR_TXBUF(adapter)->m_pkthdr.len);
              adapter->silan_bufdata.cur_tx = (entry+1) % SL_TXD_CNT;	

              PDEBUG("sln%d:Queue tx packet size %d to tx-descriptor %d.\n",adapter->silan_unit, len, entry);   
        }

        /* Tx buffer chain full */
	if (SL_CUR_TXBUF(adapter) != NULL)				
		ifp->if_flags |= IFF_OACTIVE;
         	
	/* Set a timeout in case the chip goes out to lunch */
	ifp->if_timer = 5;
       
        return;
}



/* Receive Data handler */
static void silan_rx(adapter)	
   struct silan_softc	*adapter;
{
     struct ether_header   *eh;
     struct mbuf	   *m;
     struct ifnet	   *ifp;
     u_int32_t		   rxstat = 0;
     u_int32_t		   rx_offset;
     caddr_t               rx_bufpos = NULL;
     u_int32_t		   cur_rx = 0;
     u_int32_t             dirty_rx;
     long                  rx_len;
     unsigned long         rx_space;
     unsigned long         rx_size = 0;
     unsigned long         rx_size_align = 0;
     u_int32_t		   rx_bytes = 0;
     unsigned long         pkt_size = 0;
     int                   s;
     

     ifp = &adapter->arpcom.ac_if;
     cur_rx = SILAN_READ_4(adapter, SL_RBW_PTR);
     dirty_rx = adapter->silan_bufdata.dirty_rx;
      		
     /* cur_rx is only 17 bits in the RxBufWPtr register. 
      * if cur_rx can be used in physical space,
      * we need to change it to 32 bits physical address 
      */
     cur_rx |= vtophys(adapter->silan_bufdata.silan_rx_buf) & (~(u_long)(SL_RX_BUFLEN - 1));
        
     if (cur_rx < vtophys(adapter->silan_bufdata.silan_rx_buf))
          cur_rx += SL_RX_BUFLEN;
     
     if (cur_rx >= dirty_rx)
          rx_len = (long)(cur_rx - dirty_rx);
     else
          rx_len = SL_RX_BUFLEN - (long)(dirty_rx - cur_rx);

     if ((rx_len > SL_RX_BUFLEN )||(rx_len < 0)){
           printf("rx len is fail\n");
           return;
     }

     if (rx_len == 0) 
           return;
    
     rx_offset = (dirty_rx - vtophys(adapter->silan_bufdata.silan_rx_buf)) & (u_long)(SL_RX_BUFLEN - 1);
	
     s = splimp();
	
     while (rx_len > 0) {
          rx_bufpos = adapter->silan_bufdata.silan_rx_buf + rx_offset;
	  rxstat = *(u_int32_t *)rx_bufpos;
          rx_size =( rxstat >> 20 )& 0x0FFF;
          rx_size_align = (rx_size + 3) & ~3;         /* for 4 bytes aligned */
          pkt_size = rx_size - ETHER_CRC_LEN;   /* Omit the four octet CRC from the length.*/
         
          PDEBUG("rx len: %ld  rx frame size:%ld  rx state:0x%x\n",rx_len, rx_size,rxstat);

	  /* errors receive packets caculatation */	
	  if (rxstat == 0 || rx_size < 16 ||!(rxstat & SL_RXSTAT_RXOK)) {
                 ifp->if_ierrors++;
                
                 if (!(rxstat & SL_RXSTAT_RXOK))
                     printf("receiver ok error\n");

                 if (!(rxstat & SL_RXSTAT_CRCOK))
		     printf("crc error\n");

		 if (rxstat & SL_RXSTAT_ALIGNERR)
		     printf("frame alignment error\n");

		 if (rxstat & (SL_RXSTAT_HUGEFRM | SL_RXSTAT_SMALLFRM))
		     printf("received frame length is error\n");
               
		 break;
	  }          
          
          rx_len -=(long)(rx_size_align + 4);  /* 4 bytes for receive frame head */
                        
          if (rx_len < 0) {
              printf("rx packets len is too small\n");
              break;
          }

#ifdef SILAN_PDEBUG
          caddr_t   p=NULL;

          printf("sln%d:rx frame content\n", adapter->silan_unit);
          p=rx_bufpos;
          for(i=0; i<30; i++,p++){
	      if(i%10 == 0) printf("\n");
              printf("%x  ",(u_char)*p);
          }
          printf("\n"); 
#endif
	  /* No errors; receive the packet. */	
	  rx_bytes = rx_bytes + rx_size + 4;   /* 4 bytes for receive frame header*/
         	
	  if (rx_bufpos == (adapter->silan_bufdata.silan_rx_buf + SL_RX_BUFLEN))
	          rx_bufpos = adapter->silan_bufdata.silan_rx_buf;
	  
          rx_bufpos = rx_bufpos + 4; /* 4 bytes for receive frame header*/        
	  rx_space = (unsigned long)((adapter->silan_bufdata.silan_rx_buf + SL_RX_BUFLEN) - rx_bufpos);
	  
	  if ( pkt_size > rx_space) {
                  m = m_devget(rx_bufpos-2, pkt_size+2, 0, ifp, NULL);   /* 2 for etherer head align */

		  if (m == NULL) {
			ifp->if_ierrors++;
			printf("sln%d: out of mbufs, tried to copy %ld bytes\n" , adapter->silan_unit, rx_space);
                  } else {
		        m_adj(m,2);
		        m_copyback(m, rx_space, pkt_size-rx_space, adapter->silan_bufdata.silan_rx_buf);
		  }
          } else {
		  m = m_devget(rx_bufpos-2, pkt_size+2, 0, ifp, NULL);

		  if (m == NULL) {
			ifp->if_ierrors++;
			printf("sln%d: out of mbufs, tried to copy %ld bytes\n",adapter->silan_unit, pkt_size);
                        printf("ierrors = %ld\n",ifp->if_ierrors);
			
		  } else {
		    m_adj(m,2);
                  }
	  }

	  ifp->if_ipackets++;
          PDEBUG("ipackers = %ld\n",ifp->if_ipackets);

          eh = mtod(m, struct ether_header *);
			 
	  /* Remove header from mbuf and pass it on. */
	  m_adj(m, sizeof(struct ether_header));
	  ether_input(ifp, eh, m);
	  
	  rx_offset =(rx_offset + rx_size + 4) & (u_long)(SL_RX_BUFLEN - 1);   /* 4 bytes for receive frame head */   
     } 
     
     adapter->silan_bufdata.dirty_rx = cur_rx;

     splx(s);

     SILAN_WRITE_4(adapter, SL_RBR_PTR, cur_rx);
 
     return;
}




/* Transmit OK/ERR handler */
static void silan_tx_intr(adapter)	
   struct silan_softc	*adapter;
{
      struct ifnet      *ifp;
      u_int32_t	        txstat;
      int               entry;

      ifp = &adapter->arpcom.ac_if;
      
      do {
   	   entry = adapter->silan_bufdata.dirty_tx; 
	   txstat = SILAN_READ_4(adapter, SL_TSD0 + entry * 4);
	   
	   if (!(txstat & (SL_TXSD_TOK | SL_TXSD_TUN |SL_TXSD_TABT)))
	          break;           /* It still hasn't been sent */

	   if ( SL_DIRTY_TXBUF(adapter) != NULL) {/*SL_DIRTY_TXBUF(x) = x->silan_bufdata.silan_tx_buf[x->silan_bufdata.dirty_tx]*/
	       m_freem(SL_DIRTY_TXBUF(adapter));	
	       SL_DIRTY_TXBUF(adapter) = NULL;
	   }
	
	   if (txstat & SL_TXSD_TOK){		
	       ifp->if_opackets++;
	       ifp->if_obytes += txstat & SL_TXSD_LENMASK;
               PDEBUG("opackets = %ld\n",ifp->if_opackets);
               ifp->if_collisions += (txstat & SL_TXSD_NCC) >> 22;	
           } else {
               ifp->if_oerrors++;
               if ((txstat & (SL_TXSD_TABT | SL_TXSD_OWC))) {
                     adapter->txcfg = TX_CFG_DEFAULT;

                     if (adapter->txenablepad)
                          adapter->txcfg |= 0x20000000;

		    SILAN_WRITE_4(adapter, SL_TX_CONFIG, adapter->txcfg);
	       }     

           }
           PDEBUG("tx done descriprtor %x\n", entry);			
	   adapter->silan_bufdata.dirty_tx = (entry +1) % SL_TXD_CNT;
	   
	   ifp->if_flags &= ~IFF_OACTIVE;
     } while(adapter->silan_bufdata.dirty_tx != adapter->silan_bufdata.cur_tx);

     if (adapter->silan_bufdata.dirty_tx == adapter->silan_bufdata.cur_tx) 
	      ifp->if_timer = 0;
     else
              ifp->if_timer = 5;
     
    return;
}


static void silan_media_intr(adapter)
     struct silan_softc     *adapter;
{
    unsigned long        phys[2];
    struct ifmedia       *ifm;
    struct ifnet         *ifp;
    int                  s;
        
    ifp =&adapter->arpcom.ac_if;
    ifm =&adapter->ifmedia;

    phys[0] = SL_MII_STAT;
    silan_mii_cmd(adapter, SL_MII0_READ, phys);

    PDEBUG("mii_stat:0x%lx\n",phys[1]);

    if (0 == (phys[1] & SL_MIISTAT_LINK)){
          printf("media is unconnect,linked down,or uncompatible\n");
          adapter->connect = false;
          silan_mii_cmd(adapter, SL_MII0_SCAN, phys);
          /* disable tx/rx */
          adapter->txcfg &= ~SL_TXCFG_EN;
          adapter->rxcfg &= ~SL_RXCFG_EN;
          SILAN_WRITE_4(adapter, SL_TX_CONFIG, adapter->txcfg );
          SILAN_WRITE_4(adapter, SL_RX_CONFIG, adapter->rxcfg );
         
	  return;
    }

     /* Link is good. Report modes and set duplex mode. */
     PDEBUG("media is connecting---> ");
     adapter->connect = true;
     s = splimp();

     phys[0] = SL_MII_STAT_OUTPUT;
     silan_mii_cmd(adapter, SL_MII0_READ, phys); 
     adapter->media_duplex = ((phys[1] & 0x0004)== 0)? IFM_HDX :IFM_FDX;
     adapter->media_speed = ((phys[1] & 0x0002)== 0) ? IFM_10_T:IFM_100_TX;

     printf("sln%d media option:%dM %s-duplex\n",
	     adapter->silan_unit,
	     adapter->media_speed == 0x6 ? 100:10,
             adapter->media_duplex == 0x100000 ? "full":"half");

     silan_mii_cmd(adapter, SL_MII0_SCAN, phys);

     silan_mac_cfg(adapter);     

     /* Enable tx/rx */
     adapter->rxcfg |= SL_RXCFG_EN;
     adapter->txcfg |= SL_TXCFG_EN;
     SILAN_WRITE_4(adapter, SL_TX_CONFIG, adapter->txcfg);
     SILAN_WRITE_4(adapter, SL_RX_CONFIG, adapter->rxcfg);
     
     splx(s);
     return;
}


/* Interrupt Handler */
static void silan_interrupt(arg)	
  void	*arg;
{
    struct silan_softc	  *adapter;
    struct ifnet	  *ifp;
    u_int32_t		  int_status;

    adapter = arg;
    ifp = &adapter->arpcom.ac_if;
    
    if(adapter->suspended)
         return;

    /* Disable interrupts. */
    SILAN_WRITE_4(adapter, SL_INT_MASK, 0);

    int_status = SILAN_READ_4(adapter, SL_INT_STATUS);
	
    if ((int_status == 0xffffffff) || (int_status & SL_INRTS) == 0)
	     return;
	     
    int_status = int_status & SL_INRTS;
    PDEBUG("int_status = 0x%x\n",int_status);

    while (0!= int_status) {	
         if (int_status & SL_INT_ROK)
	      silan_rx(adapter);

         if (int_status & SL_INT_TOK)
              silan_tx_intr(adapter);
        
         if (int_status & SL_INT_RBO){
	       ifp->if_ierrors++;
               PDEBUG("rx buffer is overflow\n");
         }

         if (int_status & (SL_INT_LINKFAIL | SL_INT_LINKOK )) {
	      silan_media_intr(adapter);	
         }      
	            
         int_status = SILAN_READ_4(adapter, SL_INT_STATUS);		
    }	

    /* Re-enable interrupts. */
    SILAN_WRITE_4(adapter, SL_INT_MASK, SL_INRTS);
    
    /* Data in Tx buffer waiting for transimission */
    if (ifp->if_snd.ifq_head != NULL){
        silan_tx(ifp);
    }
    
    return;
}


static void  silan_tick(x)
        void   *x;
{
       struct silan_softc      *adapter = x;
                    
       adapter->silan_state = timeout(silan_tick, adapter, hz);

       return;
}


static int silan_ioctl(ifp, command, data)
	struct ifnet		*ifp;
	unsigned long		command;
	caddr_t			data;
{
	struct silan_softc	*adapter = ifp->if_softc;
        struct ifreq            *ifr = (struct ifreq*)data;
	int			s;
        int                     error = 0;
         
	s = splimp();     
	switch(command) {
	case SIOCSIFADDR:
	case SIOCGIFADDR:
	case SIOCSIFMTU:
		error = ether_ioctl(ifp, command, data);
		break;
	case SIOCSIFFLAGS:
	  	if (ifp->if_flags & IFF_UP) {
		    silan_init(adapter);
		} else {
	            if (ifp->if_flags & IFF_RUNNING)
			silan_stop(adapter);
		}
		error = 0;
                break;
	case SIOCADDMULTI:
	case SIOCDELMULTI:
	       silan_set_multi(adapter);
               error = 0;
	       break;
	case SIOCGIFMEDIA:
	case SIOCSIFMEDIA:
	       error = ifmedia_ioctl(ifp,ifr,&adapter->ifmedia, command);
               break; 
	default:
		error = EINVAL;
		break;
	}

	(void)splx(s);

	return(error);
}



static void silan_watchdog(ifp)
   struct ifnet	  *ifp;
{
    struct silan_softc	  *adapter;

    adapter = ifp->if_softc;
    ifp->if_timer = 0;
        
    printf("sln%d: watchdog\n", adapter->silan_unit);
    ifp->if_oerrors++;
    
    silan_tx_intr(adapter);
    silan_rx(adapter);
    silan_stop(adapter);
    silan_init(adapter);
    
    if (ifp->if_snd.ifq_head != NULL)
	  silan_tx(ifp);
    
    return;
}



/* Stop all chip I/O */
static void silan_shutdown(dev)	
   device_t    dev;
{
   struct silan_softc   *adapter;

   adapter = device_get_softc(dev);
   
   silan_stop(adapter);
   
   return;
}



/* device suspend routine */
static int silan_suspend (dev)
     device_t     dev;
{ 
  struct silan_softc    *adapter;

  adapter = device_get_softc(dev);
  silan_stop(adapter);
  adapter->suspended = true;
  
  return(0);
}


/* device resume routine */
static int silan_resume (dev)
     device_t     dev;
{
  struct silan_softc     *adapter;
  struct ifnet           *ifp  ;

  adapter = device_get_softc(dev);
  ifp = &adapter->arpcom.ac_if;
  
  pci_enable_busmaster(dev);
  pci_enable_io(dev,SL_RES);

  if (ifp->if_flags & IFF_UP)
    silan_init(adapter);

  adapter->suspended = false;
 
  return (0);

}


